function [out,fmin,parout]=comirror_formd(Ffun,Ffun_sgrad,Gfun,Gfun_sgrad,startx,feas_eps,simR,maxiter,isequal,lowb,upb,ecoflag,printflag,printcon,stuckiter)
% COMIRROR_FORMD employs the co-mirror descent method inside the md software. 
% It is used when the underlying set is a variant of the unit simplex set.
% ====================================================
% Usage:
% out               = COMIRROR_FORMD(Ffun,Ffun_sgrad,Gfun,Gfun_sgrad,startx,feas_eps,simR,maxiter,isequal,lowb,upb,ecoflag,printflag,printcon,stuckiter)
% [out,fmin]        = COMIRROR_FORMD(Ffun,Ffun_sgrad,Gfun,Gfun_sgrad,startx,feas_eps,simR,maxiter,isequal,lowb,upb,ecoflag,printflag,printcon,stuckiter)
% [out,fmin,parout] = COMIRROR_FORMD(Ffun,Ffun_sgrad,Gfun,Gfun_sgrad,startx,feas_eps,simR,maxiter,isequal,lowb,upb,ecoflag,printflag,printcon,stuckiter)
% ====================================================
% Input:
% Ffun        - function handle for the objective function f
% Ffun_sgrad  - function handle for the subgradient of the objective function f
% Gfun        - function handle for the constraint function g
% Gfun_sgrad  - function handle for the subgradients of the components of the constraint function (arranged in columns)
% startx      - a starting vector
% feas_eps    - feasibility tolerance
% simR        - the radius of the simplex
% maxiter     - maximal number of iterations
% isequal     - true if equality is imposed, otherwise false
% lowb        - lower bound on the variables
% upb         - upper bound on the variables
% ecoflag     - true if economic version (without calculating objective function
%                              values) should run, otherwise false
% printflag   - true if internal printing should take place, otherwise false
% printcon    - true if constraint value should be printed, otherwise false
% stuckiter   - maximal allowed number of iterations without improvement
% ====================================================
% Output:
% out    - optimal solution (up to a tolerance)
% fmin   - optimal value (up to a tolerance)
% parout - a struct containing additional information related to the convergence. The fields of parout are:
%    iterNum   - number of performed iterations
%    funValVec - vector of all function values generated by the method
%    consVec   - vector containing all the constraints violations quantities (when relevant)

% This file is part of the FOM package - a collection of first order methods for solving convex optimization problems
% Copyright (C) 2017 Amir and Nili Beck
% 
% This program is free software: you can redistribute it and/or modify
% it under the terms of the GNU General Public License as published by
% the Free Software Foundation, either version 3 of the License, or
% (at your option) any later version.
% 
% This program is distributed in the hope that it will be useful,
% but WITHOUT ANY WARRANTY; without even the implied warranty of
% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
% GNU General Public License for more details.
% 
% You should have received a copy of the GNU General Public License
% along with this program.  If not, see <http://www.gnu.org/licenses/>.

%constant values:
delta=1e-13;
alpha=1;
n=length(startx) ;
theta=log(n);
startfmin = inf ;

%starting values
iternu=0; %iteration number
newx = startx ;
out = startx ; %best x so far

%moving the start vector into simplex 1 with lowb=0

if (isscalar(lowb) )
    sumlowb = n * lowb ; %sumlowb represents the sum of the lowb vector
else
    sumlowb = sum(lowb) ;
end

%changing the upb, to suit the change that will be made in the vector
upb = (upb - lowb) /(simR - sumlowb) ;
%Make our initial vector feasible, satisfying lowb and upb
if (isequal)
    newx = proj_hyperplane_box(newx,ones(n,1),simR,lowb,upb) ;
else
    newx = proj_halfspace_box(newx,ones(n,1),simR,lowb,upb) ;
end

%Move our solution vector to the 0,1 simplex
newx=(newx-lowb)/(simR-sumlowb) ;

derf= CoverF_objder(Ffun_sgrad,newx,lowb,simR,sumlowb) ;
[newg,derg] = CoverF_con(Gfun,Gfun_sgrad,newx,lowb,simR,sumlowb) ;
fmin = startfmin ;
savediter=-1; % to indicate that we haven't found any feasible solution so far
%starting parout, when necessary
if (nargout  == 3)
    parout = struct ;
    parout.iterNum = 0 ;
    parout.funValVec = [] ;
    if (printcon)
        parout.consVec  = [] ;
    end
end

%the iterations
while( (iternu < maxiter))
    iternu=iternu+1;
    xiter=newx ;
    if (newg < feas_eps)
        ftag = derf; %subgradient for objective function
    else
        ftag = derg ;%subgradient for constraint function
    end
    gama = sqrt(theta*alpha)/(norm(ftag,inf)*sqrt(iternu));
    xiter = xiter + delta * (xiter == 0) ; %before the log, making sure there are no
    %zeros in xiter
    omega=log(xiter)+1;
    a=gama*ftag-omega;
    
    %checking whether the sum is smaller than simR when it is necessary
    p=-a ;
    newx = exp(p-1) ;
    %we are using coverF, simR here is exchanged for 1
    if ((sum(newx) > 1) || (isequal == true))
        phat=   p-max(p) ;
        ep=exp(phat) ;
        newx = 1*ep/sum(ep) ;
    end
    
    if (~all(newx < upb))
        newx =find_newx_under_ub(a,upb,isequal) ;
    end
    
    derf = CoverF_objder(Ffun_sgrad,newx,lowb,simR,sumlowb) ;
    [newg,derg] = CoverF_con(Gfun,Gfun_sgrad,newx,lowb,simR,sumlowb) ;
    
    if (~ecoflag)
        
        if (iternu - savediter > stuckiter)
            fprintf('Stopping because of %d iterations with no improvement\n',stuckiter) ;
            break
        end
        newf = CoverF_objval(Ffun,newx,lowb,simR,sumlowb) ;
        
        if (nargout  == 3)
            %updating parout
            parout.funValVec = [ parout.funValVec ; newf] ;
            if (printcon)
                parout.consVec = [ parout.consVec ; newg] ;
            end
        end
        
        if ( (newf < fmin ) && (newg < feas_eps)  )
            fmin = newf ;
            out = newx ;
            if (printflag)
                if (printcon)
                    fprintf('%6d \t    %12f\t %12f\n',iternu,fmin,newg);
                else
                    fprintf('%6d \t    %12f \n',iternu,fmin);
                end
            end
            savediter=iternu ;
        end
    else %ecoflag = true, looking for a feasible solution
        if (newg < feas_eps)
            out = newx ;
            savediter=iternu ;
        end
    end
    
end

if (savediter ~= -1)
    %found a feasible solution
    %moving the result vector back
    out = (simR-sumlowb)*out+lowb ;
    
else %never set a value to savediter, no feasible solution found
    fprintf('No feasible solution found\n') ;
    out = zeros(size(startx)) ;
    fmin = startfmin ;
end

if (nargout == 3)
    %updating parout.iterNum
    parout.iterNum = iternu ;
end

    function val=CoverF_objval(fun,x,lowb,sumx,sumlowb)
        %to be used for the objective function value
        val = fun((sumx-sumlowb)*x+lowb);
        
    end


    function der=CoverF_objder(fun_sgrad,x,lowb,sumx,sumlowb)
        %to be used for the objective function subgradient
        der = fun_sgrad((sumx-sumlowb)*x+lowb);
        der = der*(sumx-sumlowb) ;
           
    end

    function [val,der]=CoverF_con(fun,fun_sgrad,x,lowb,sumx,sumlowb)
        %to be used for the constrains
        fun_val = fun((sumx-sumlowb)*x+lowb);
        fun_der = fun_sgrad((sumx-sumlowb)*x+lowb);
        fun_der = fun_der*(sumx-sumlowb) ;
        
        %these lines handle the case when the constraint function returns
        %a vector or a matrix, because there is more than one constraint
        [val,pl] = max(fun_val) ;
        der = fun_der(:,pl) ;
    end
        

end

